<?php

namespace App\Http\Controllers;

use App\Models\Receipt;
use App\Models\ShareLink;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ShareLinkController extends Controller
{
    public function store(Request $request, Receipt $receipt)
    {
        $this->authorize('view', $receipt);

        $data = $request->validate([
            'expires_at' => 'nullable|date',
            'max_views' => 'nullable|integer|min:1|max:1000',
        ]);

        $link = ShareLink::create([
            'receipt_id' => $receipt->id,
            'token' => hash('sha256', Str::random(80)),
            'expires_at' => $data['expires_at'] ?? null,
            'max_views' => $data['max_views'] ?? null,
            'view_count' => 0,
        ]);

        return back()->with('status', 'Proof link created.')
            ->with('proof_url', route('proof.show', $link->token));
    }

    public function show(string $token)
    {
        $link = ShareLink::where('token', $token)->firstOrFail();

        if ($link->expires_at && now()->greaterThan($link->expires_at)) {
            abort(410, 'Link expired');
        }

        if ($link->max_views !== null && $link->view_count >= $link->max_views) {
            abort(410, 'Link expired');
        }

        $link->increment('view_count');

        $receipt = $link->receipt;

        return view('proof.show', compact('receipt', 'link'));
    }
}
