<?php

namespace App\Http\Controllers;

use App\Models\Receipt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ReceiptController extends Controller
{
    public function index(Request $request)
    {
        $q = Receipt::query()->where('user_id', $request->user()->id);

        $search = trim((string)$request->get('q'));
        if ($search !== '') {
            $q->where(function ($w) use ($search) {
                $w->where('merchant', 'like', "%{$search}%")
                  ->orWhere('original_filename', 'like', "%{$search}%")
                  ->orWhere('notes', 'like', "%{$search}%");
            });
        }

        return view('receipts.index', [
            'receipts' => $q->latest()->paginate(20)->withQueryString(),
            'search' => $search,
        ]);
    }

    public function create()
    {
        return view('receipts.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'file' => 'required|file|max:10240|mimes:pdf,jpg,jpeg,png,webp',
            'merchant' => 'nullable|string|max:255',
            'total_amount' => 'nullable|numeric',
            'currency' => 'nullable|string|size:3',
            'purchased_at' => 'nullable|date',
            'notes' => 'nullable|string|max:5000',
        ]);

        $file = $request->file('file');
        $hash = hash_file('sha256', $file->getRealPath());
        $path = $file->store('private/receipts', ['disk' => 'local']);

        $receipt = Receipt::create([
            'user_id' => $request->user()->id,
            'original_filename' => $file->getClientOriginalName(),
            'stored_path' => $path,
            'mime_type' => (string)$file->getClientMimeType(),
            'size_bytes' => (int)$file->getSize(),
            'sha256' => $hash,
            'merchant' => $data['merchant'] ?? null,
            'total_amount' => $data['total_amount'] ?? null,
            'currency' => strtoupper($data['currency'] ?? 'CAD'),
            'purchased_at' => $data['purchased_at'] ?? null,
            'notes' => $data['notes'] ?? null,
        ]);

        return redirect()->route('receipts.show', $receipt);
    }

    public function show(Receipt $receipt)
    {
        $this->authorize('view', $receipt);
        return view('receipts.show', compact('receipt'));
    }

    public function destroy(Receipt $receipt)
    {
        $this->authorize('delete', $receipt);

        Storage::disk('local')->delete($receipt->stored_path);
        $receipt->delete();

        return redirect()->route('receipts.index')->with('status', 'Deleted.');
    }
}
