<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Receipt;
use App\Models\ShareLink;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ShareLinkApiController extends Controller
{
    public function store(Request $r, Receipt $receipt)
    {
        $this->authorize('view', $receipt);

        $data = $r->validate([
            'expires_at' => 'nullable|date',
            'max_views' => 'nullable|integer|min:1|max:1000',
        ]);

        $link = ShareLink::create([
            'receipt_id' => $receipt->id,
            'token' => hash('sha256', Str::random(80)),
            'expires_at' => $data['expires_at'] ?? null,
            'max_views' => $data['max_views'] ?? null,
        ]);

        return response()->json([
            'token' => $link->token,
            'url' => url('/p/' . $link->token),
        ], 201);
    }

    public function show(string $token)
    {
        $link = ShareLink::where('token', $token)->firstOrFail();

        if ($link->expires_at && now()->gt($link->expires_at)) {
            return response()->json(['message' => 'Expired'], 410);
        }
        if ($link->max_views && $link->view_count >= $link->max_views) {
            return response()->json(['message' => 'Expired'], 410);
        }

        $link->increment('view_count');
        $receipt = $link->receipt;

        return response()->json([
            'receipt' => [
                'merchant' => $receipt->merchant,
                'total_amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'purchased_at' => optional($receipt->purchased_at)->toISOString(),
                'notes' => $receipt->notes,
                'original_filename' => $receipt->original_filename,
            ],
            'link' => [
                'expires_at' => optional($link->expires_at)->toISOString(),
                'max_views' => $link->max_views,
                'view_count' => $link->view_count,
            ],
        ]);
    }
}
