<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Receipt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ReceiptApiController extends Controller
{
    public function index(Request $r)
    {
        $q = Receipt::where('user_id', $r->user()->id);

        if ($s = trim((string)$r->get('q'))) {
            $q->where(function ($w) use ($s) {
                $w->where('merchant','like',"%{$s}%")
                  ->orWhere('original_filename','like',"%{$s}%")
                  ->orWhere('notes','like',"%{$s}%");
            });
        }

        return response()->json($q->latest()->paginate(20));
    }

    public function store(Request $r)
    {
        $r->validate([
            'file' => 'required|file|max:10240|mimes:pdf,jpg,jpeg,png,webp',
            'merchant' => 'nullable|string|max:255',
            'total_amount' => 'nullable|numeric',
            'currency' => 'nullable|string|size:3',
            'purchased_at' => 'nullable|date',
            'notes' => 'nullable|string|max:5000',
        ]);

        $file = $r->file('file');
        $hash = hash_file('sha256', $file->getRealPath());
        $path = $file->store('private/receipts', ['disk' => 'local']);

        $receipt = Receipt::create([
            'user_id' => $r->user()->id,
            'original_filename' => $file->getClientOriginalName(),
            'stored_path' => $path,
            'mime_type' => $file->getClientMimeType(),
            'size_bytes' => $file->getSize(),
            'sha256' => $hash,
            'merchant' => $r->merchant,
            'total_amount' => $r->total_amount,
            'currency' => strtoupper($r->currency ?? 'CAD'),
            'purchased_at' => $r->purchased_at,
            'notes' => $r->notes,
        ]);

        return response()->json($receipt, 201);
    }

    public function show(Request $r, Receipt $receipt)
    {
        $this->authorize('view', $receipt);
        return response()->json($receipt);
    }

    public function download(Request $r, Receipt $receipt)
    {
        $this->authorize('view', $receipt);

        return Storage::disk('local')->download(
            $receipt->stored_path,
            $receipt->original_filename
        );
    }
}
